/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib/gi18n.h>

#include "launcher-iconview.h"

#include <string.h>
#include <tidy/tidy.h>

#include "clutter-drag-server.h"
#include "clutter-focus-group.h"
#include "clutter-focus-manager.h"
#include "launcher-app.h"
#include "launcher-add.h"
#include "launcher-behave.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-icon.h"
#include "launcher-shortcut-editor.h"
#include "launcher-util.h"

static void clutter_focus_group_iface_init (ClutterFocusGroupIface *iface);

G_DEFINE_TYPE_WITH_CODE (LauncherIconview,
                         launcher_iconview, 
                         TIDY_TYPE_VIEWPORT,
                         G_IMPLEMENT_INTERFACE (CLUTTER_TYPE_FOCUS_GROUP,
                                              clutter_focus_group_iface_init));

#define LAUNCHER_ICONVIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ICONVIEW, \
  LauncherIconviewPrivate))

#define BG_PADDING ((cfg->bg_padding))

static ClutterActor *iconviewbg      = NULL;
static GdkPixbuf    *iconviewbg_p    = NULL;
static ClutterActor *shadowbg        = NULL;
static GList *icons = NULL;
static gint n_icons = 0;

struct _LauncherIconviewPrivate
{
  LauncherMenuCategory *cat;
  GList *icons;

  ClutterActor *bg;
  ClutterActor *grid;
  ClutterActor *scroll;
  ClutterActor *add;
  ClutterActor *shadow;

  TidyAdjustment *adjust;

  guint count;
  guint tag;

  ClutterActor *drag_icon;
  ClutterActor *drag_context;
  ClutterActor *drag_sep;
  gint old_pos;
  gint drag_xtag;
  gint drag_ytag;
  gint drag_col; 
  gint drag_row;

  ClutterActor *focused;

  LauncherIcon *last_icon;

  ClutterTimeline *time;
  ClutterEffectTemplate *temp;

  ClutterTimeline  *reorder_time;
  ClutterBehaviour *reorder_behave;
};

static gboolean
eat_events (ClutterActor *actor, ClutterEvent *event)
{
  return TRUE;
}

void    
launcher_iconview_disable (LauncherIconview   *view)
{
  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  g_signal_connect (view, "captured-event", G_CALLBACK (eat_events), NULL);
}

/*
 * DRAG SORTING 
*/
  
static void
reorder_func (ClutterBehaviour *behave,
              guint32           alpha_value,
              LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  GList *i;
  gfloat factor;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  factor = (gfloat)alpha_value / CLUTTER_ALPHA_MAX_ALPHA;

  for (i = priv->icons; i; i = i->next)
  {
    ClutterActor *actor = i->data;
    gint x, y, endx, endy, newx, newy;

    x = clutter_actor_get_x (actor);
    y = clutter_actor_get_y (actor);
    
    endx = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (actor), "end-x"));
    endy = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (actor), "end-y"));

    newx = alpha_value * (endx - x)
            / CLUTTER_ALPHA_MAX_ALPHA
            + x;
    newy = alpha_value * (endy -y)
            / CLUTTER_ALPHA_MAX_ALPHA
            + y;
    
    clutter_actor_set_position (actor, newx, newy);
  }

  clutter_actor_queue_redraw (CLUTTER_ACTOR (view));
}

static void
get_drop_cell (LauncherIconview *view, 
               gint              x, 
               gint              y,
               gint             *col,
               gint             *row)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  gint view_padding = 0;
  gint icon_width  = 0;
  gint icon_height = 0;
  gint view_width  = 0;
  gint view_height = 0;
  gint view_x      = 0;
  gint view_y      = 0;
  gint cols;
  gint rows;
  gint offset =0;
     
  priv = view->priv;

  view_padding = cfg->iconview_padding;
  icon_width = cfg->icon_width;
  icon_height = cfg->icon_height;
  view_width = CAW (priv->bg);
  view_height = CAH (priv->bg);
  clutter_actor_get_abs_position (priv->bg, &view_x, &view_y);
  
  *row = *col = -1;

  if (x < view_x || x > view_x + view_width)
    return;

  if (y < view_y || y > view_y + view_height)
    return;

  if (TIDY_IS_ADJUSTMENT (priv->adjust))
    offset = (tidy_adjustment_get_value (priv->adjust)) 
                / icon_height;
  cols = (x - view_x) / icon_width;
  rows = (y - view_y) / icon_height; 
  rows += offset;

  *col = cols;
  *row = rows;

  if (!CLUTTER_IS_ACTOR (priv->drag_sep))
    return;
}

static void
on_drag_finished (ClutterDragServer *server,
                  ClutterDragPoint  *point,
                  LauncherIconview  *view)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GList *apps = NULL, *i;
  gint col = 0, row = 0;
  gint new_pos;
  gint icon_width = cfg->icon_width;
  gint icon_height = cfg->icon_height;
  gint view_padding = cfg->iconview_padding;
  gint j=0, xoffset =0, yoffset =0, app_cols = cfg->iconview_cols;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  clutter_actor_destroy (priv->drag_sep);

  g_signal_handler_disconnect (priv->drag_context, priv->drag_xtag);
  g_signal_handler_disconnect (priv->drag_context, priv->drag_ytag);

  get_drop_cell (view, point->x, point->y, &col, &row);

  if (col == -1)
  {
    /* Restore old positions */
    priv->icons = g_list_remove (priv->icons, priv->drag_icon);
    priv->icons = g_list_insert (priv->icons, priv->drag_icon, priv->old_pos);
  }  
  else
  {
    new_pos = col + (row * cfg->iconview_cols);

    priv->icons = g_list_remove (priv->icons, priv->drag_icon);
    priv->icons = g_list_insert (priv->icons, priv->drag_icon, new_pos);
  }
  
  clutter_actor_set_opacity (priv->drag_icon, 255);
  launcher_icon_reset_after_drag (LAUNCHER_ICON (priv->drag_icon));
  
  for (i = priv->icons; i; i = i->next)
  {
    ClutterActor *actor = i->data;
    
    g_object_set_data (G_OBJECT (actor), "end-x", GINT_TO_POINTER (xoffset));
    g_object_set_data (G_OBJECT (actor), "end-y", GINT_TO_POINTER (yoffset));
        
    if (j == (app_cols -1))
    {
      j = 0;
      xoffset = 0;
      yoffset += icon_height + view_padding;
    }
    else
    {
      xoffset += icon_width + view_padding;
      j++;
    }

  }

  clutter_timeline_start (priv->reorder_time);

  for (i = priv->icons; i; i = i->next)
    apps = g_list_append (apps, launcher_icon_get_application (i->data));

  launcher_menu_set_applications (launcher_menu_get_default (),
                                  priv->cat, 
                                  apps);

  priv->drag_context = NULL;
  priv->old_pos = 0;
  priv->drag_icon = NULL;

  {
    GList *i;
    gdouble diff = 0.0;
    gint rows = 0;
    gint j = 0, k = 0;

    if (TIDY_IS_ADJUSTMENT (priv->adjust))
      diff = tidy_adjustment_get_value (priv->adjust);

    rows = diff / cfg->icon_height;

    if ((rows+2) > n_icons/6)
      rows -= 2;

    for (i = priv->icons; i; i = i->next)
    {
      if (j < (rows * 6))
        ;
      else
      {
        if (i && i->data && LAUNCHER_IS_ICON (i->data))
          launcher_icon_set_position (LAUNCHER_ICON (i->data), k);
        k++;
      }
      j++;
    }
  }
}

static void
on_context_moved (ClutterActor     *context, 
                  GParamSpec       *arg1,
                  LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GList *i;
  gint x=0, y=0;
  gint col = 0, row = 0;
  gint new_pos;
  gint icon_width = cfg->icon_width;
  gint icon_height = cfg->icon_height;
  gint view_padding = cfg->iconview_padding;
  gint j=0, xoffset =0, yoffset =0, app_cols = cfg->iconview_cols;

  priv = view->priv;
  
  clutter_actor_get_position (context, &x, &y);
  get_drop_cell (view, x, y, &col, &row);

  if (col == -1)
  {
    clutter_actor_set_opacity (priv->drag_sep, 0);
  }

  new_pos = col + (row * cfg->iconview_cols);

  priv->icons = g_list_remove (priv->icons, priv->drag_icon);
  priv->icons = g_list_insert (priv->icons, priv->drag_icon, new_pos);

  for (i = priv->icons; i; i = i->next)
  {
    ClutterActor *actor = i->data;

    if (actor == context)
      continue;
    
    g_object_set_data (G_OBJECT (actor), "end-x", GINT_TO_POINTER (xoffset));
    g_object_set_data (G_OBJECT (actor), "end-y", GINT_TO_POINTER (yoffset));
        
    if (j == (app_cols -1))
    {
      j = 0;
      xoffset = 0;
      yoffset += icon_height + view_padding;
    }
    else
    {
      xoffset += icon_width + view_padding;
      j++;
    }
  }
  clutter_timeline_start (priv->reorder_time);
}

static void
on_drag_started (ClutterDragServer *server, LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *context;
  gint icon_height = (launcher_config_get_default ())->icon_height;
  
  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  priv->drag_context = context = clutter_drag_server_get_context (server);
  if (!CLUTTER_IS_ACTOR (context))
    return;

  priv->drag_icon = g_object_get_data (G_OBJECT (context), "LauncherIcon");
  if (!LAUNCHER_IS_ICON (priv->drag_icon))
    return;

  priv->old_pos = g_list_index (priv->icons, priv->drag_icon);

  priv->drag_sep = launcher_util_texture_new_from_file (cfg->sep);
  clutter_container_add_actor (CLUTTER_CONTAINER (clutter_stage_get_default ()),
                               priv->drag_sep);
  clutter_actor_set_size (priv->drag_sep, 
                          30, 
                          icon_height + 10);
  clutter_actor_set_anchor_point (priv->drag_sep, 10, 0);
  clutter_actor_set_opacity (priv->drag_sep, 0);
  clutter_actor_show (priv->drag_sep);

  priv->drag_xtag = g_signal_connect (context, "notify::x", 
                                      G_CALLBACK (on_context_moved), view);
  priv->drag_ytag = g_signal_connect (context, "notify::y", 
                                      G_CALLBACK (on_context_moved), view);

  clutter_actor_set_opacity (priv->drag_icon, 0);
}

/*
 * </DRAG SORTING>
 */

static void
on_add_reponse (GtkWidget *dialog, gint res, LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;

  if (!LAUNCHER_IS_ICONVIEW (view))
  {
    gtk_widget_destroy (dialog);
    return;
  }
  priv = view->priv;

  if (res == GTK_RESPONSE_YES)
  {

     gchar *name = NULL, *icon = NULL, *exec = NULL;
     g_object_get (dialog, 
                   "shortcut-name", &name,
                   "shortcut-exec", &exec,
                   "shortcut-icon", &icon,
                   NULL);
     launcher_menu_add_application (launcher_menu_get_default (),
                                    priv->cat,
                                    name, exec, icon);
     g_free (name);
     g_free (icon);
     g_free (exec);
  }

  gtk_widget_destroy (dialog);
}

static gboolean
on_add_clicked (ClutterActor *actor, 
                LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  GtkWidget *dialog;

  g_return_val_if_fail (LAUNCHER_IS_ICONVIEW (view), FALSE);
  priv = view->priv;

  dialog = launcher_shortcut_editor_new ();
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                          "gtk-cancel", GTK_RESPONSE_CANCEL,
                          GTK_STOCK_APPLY, GTK_RESPONSE_YES,
                          NULL);
  g_object_set (dialog, "title", _("Add Shortcut to Category"), NULL);
  g_signal_connect (dialog, "response", G_CALLBACK (on_add_reponse), view);

  launcher_util_present_window (GTK_WINDOW (dialog));

  return TRUE;
}

gboolean
on_scroll_event (ClutterActor       *actor,
                 ClutterScrollEvent *event,
                 TidyAdjustment     *adjust)
{
  LauncherConfig *cfg = launcher_config_get_default ();
  gint VALUE = cfg->iconview_padding + cfg->icon_height;
  gdouble diff = VALUE;
  gdouble lower, upper;

  if (event->direction == CLUTTER_SCROLL_UP)
    diff = -VALUE;

  diff += tidy_adjustment_get_value (adjust)+diff;

  tidy_adjustment_get_values (adjust, NULL, &lower, &upper, NULL, NULL, NULL);

  diff = CLAMP (diff, lower, upper);

  tidy_adjustment_set_value (adjust, diff);

  {
    GList *i;
    gint rows = 0;
    gint j = 0, k = 0;

    rows = diff / cfg->icon_height;

    if ((rows+2) > n_icons/6)
      rows -= 2;

    for (i = icons; i; i = i->next)
    {
      if (j < (rows * 6))
        ;
      else
      {
        launcher_icon_set_position (LAUNCHER_ICON (i->data), k);
        k++;
      }
      j++;
    }
  }

  return TRUE;
}

static gboolean
on_up_pressed (ClutterActor       *actor,
               ClutterButtonEvent *event,
               TidyAdjustment     *adjust)
{
  LauncherConfig *cfg = launcher_config_get_default ();
  gint VALUE = cfg->iconview_padding + cfg->icon_height;
  gdouble diff = -VALUE;
  gdouble lower, upper;

  diff += tidy_adjustment_get_value (adjust)+diff;

  tidy_adjustment_get_values (adjust, NULL, &lower, &upper, NULL, NULL, NULL);

  diff = CLAMP (diff, lower, upper);

  tidy_adjustment_set_value (adjust, diff);

  return TRUE;
}

static gboolean
on_down_pressed (ClutterActor       *actor,
                 ClutterButtonEvent *event,
                 TidyAdjustment     *adjust)
{
  LauncherConfig *cfg = launcher_config_get_default ();
  gint VALUE = cfg->iconview_padding + cfg->icon_height;
  gdouble diff = VALUE;
  gdouble lower, upper;

  diff += tidy_adjustment_get_value (adjust)+diff;

  tidy_adjustment_get_values (adjust, NULL, &lower, &upper, NULL, NULL, NULL);

  diff = CLAMP (diff, lower, upper);

  tidy_adjustment_set_value (adjust, diff);

  return TRUE;
}  

gboolean
launcher_iconview_scroll_event (ClutterActor       *actor,
                                ClutterScrollEvent *event,
                                LauncherIconview   *port)
{
  TidyAdjustment *adjust;

  if (!TIDY_IS_VIEWPORT (port)) 
    return FALSE;

  if (!CLUTTER_IS_ACTOR (port->priv->scroll))
    return FALSE;
  
  tidy_scrollable_get_adjustments (TIDY_SCROLLABLE (port->priv->grid), 
                                   NULL, &adjust);

  return on_scroll_event (actor, event, adjust);
}

static gboolean
foldout (LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  gint width = cfg->icon_width;
  gint i, length;

  g_return_val_if_fail (LAUNCHER_IS_ICONVIEW (view), FALSE);
  priv = view->priv;

  if (!CLUTTER_IS_ACTOR (priv->bg) || !CLUTTER_IS_ACTOR (priv->grid))
    return FALSE;

  length = cfg->iconview_rows * cfg->iconview_cols;
  for (i = priv->count-1; i < length; i += cfg->iconview_cols)
  {
    LauncherIcon *icon = g_list_nth_data (priv->icons, i);
    if (LAUNCHER_IS_ICON (icon))
    {
      LauncherMenuApplication *app;
      app= (LauncherMenuApplication*)g_object_get_data (G_OBJECT (icon), "app");
      
      launcher_icon_set_application (icon, app);
      launcher_icon_set_position (icon, i);
    }
  }
  
  width *= priv->count;

  clutter_actor_set_clip (priv->bg, -BG_PADDING, -BG_PADDING, 
                          width + BG_PADDING*2,
                          CAH (priv->bg)+ 20);
  clutter_actor_set_clip (priv->shadow, -BG_PADDING-4, -BG_PADDING-4, 
                          width + BG_PADDING*2+8,
                          CAH (priv->bg)+ 20 + 8);
  clutter_actor_set_clip (priv->grid, 
                          0, 
                          0,
                          width,
                          cfg->iconview_height);

  clutter_actor_queue_redraw (priv->bg);
  clutter_actor_queue_redraw (priv->grid);

  priv->count++;

  if (priv->count > cfg->iconview_cols)
  {
    length = g_list_length (priv->icons);
    for (i = cfg->iconview_rows *cfg->iconview_cols; 
         i < length; i++)
    {
      LauncherIcon *icon = g_list_nth_data (priv->icons, i);
      if (LAUNCHER_IS_ICON (icon))
      {
        LauncherMenuApplication *app;
        app = (LauncherMenuApplication*)g_object_get_data (G_OBJECT (icon), 
                                                           "app");
      
        launcher_icon_set_application (icon, app);
        launcher_icon_set_position (icon, i);
      }
    }
     
    if (CLUTTER_IS_ACTOR (priv->scroll))
      clutter_effect_fade (priv->temp, priv->scroll, 255, NULL, NULL);

    clutter_actor_remove_clip (priv->bg);
    clutter_actor_remove_clip (priv->shadow);    
    launcher_util_start_input ();
    return FALSE;
  }

  return TRUE;
}

static void
on_icon_flipped (LauncherIcon *flipped, LauncherIconview *view)
{
  LauncherIconviewPrivate *priv;
  GList *i;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  priv = view->priv;

  for (i = priv->icons; i; i = i->next)
  {
    LauncherIcon *icon = i->data;

    if (icon != flipped)
      launcher_icon_set_flipped (icon, FALSE);
  }
}

static gboolean
on_enter (LauncherIcon         *icon, 
          ClutterCrossingEvent *event, 
          LauncherIconview     *view)
{
  LauncherIconviewPrivate *priv;

  if (!LAUNCHER_IS_ICONVIEW (view) || !LAUNCHER_IS_ICON (icon))
    return FALSE;
  priv = view->priv;
  
  launcher_icon_set_show_target (icon, event, TRUE);
  if (LAUNCHER_IS_ICON (priv->last_icon))
    launcher_icon_set_show_target (priv->last_icon, event, FALSE);

  priv->last_icon = icon;

  return FALSE;
}

static gboolean
on_leave (LauncherIcon         *icon, 
          ClutterCrossingEvent *event, 
          LauncherIconview     *view)
{
  LauncherIconviewPrivate *priv;

  if (!LAUNCHER_IS_ICONVIEW (view) || !LAUNCHER_IS_ICON (icon))
    return FALSE;
  priv = view->priv;
  
  launcher_icon_set_show_target (icon, event, FALSE);
  if (LAUNCHER_IS_ICON (priv->last_icon))
    launcher_icon_set_show_target (priv->last_icon, event, FALSE);

  return FALSE;
}

static gboolean
eat_click (ClutterActor *actor, ClutterButtonEvent *event)
{
  return TRUE;
}

void           
launcher_iconview_set_category (LauncherIconview     *view,
                                LauncherMenuCategory *cat,
                                gboolean              first_run)
{
  LauncherIconviewPrivate *priv;
  LauncherMenu *menu = launcher_menu_get_default ();
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *scroll, *tex, *frame, *line, *button, *shadow;
  ClutterActor *stage = clutter_stage_get_default ();
  ClutterFocusManager *manager = CLUTTER_FOCUS_MANAGER (clutter_focus_manager_get_default ());
  TidyAdjustment *adjustment;
  GList *l;
  gint xoffset, yoffset, i = 0, rows = 0, j = 0;
  gint app_cols = cfg->iconview_cols;
  gint view_width = cfg->iconview_width;
  gint view_height = cfg->iconview_height;
  gint icon_height = cfg->icon_height;
  gint cat_height = cfg->cat_height;

  g_return_if_fail (LAUNCHER_IS_ICONVIEW (view));
  g_return_if_fail (cat);
  priv = view->priv;

  if (!launcher_menu_get_n_categories (menu))
    return;

  shadow = launcher_util_texture_new_from_file (PKGDATADIR"/shadow.svg");
  priv->shadow = shadow;
  clutter_container_add_actor (CLUTTER_CONTAINER (view), shadow);
  clutter_actor_set_size (shadow, view_width+8+(2*BG_PADDING), 
                          view_height+8+(2*BG_PADDING));
  clutter_actor_set_position (shadow, -4-BG_PADDING, -4-BG_PADDING);
  clutter_actor_set_opacity (shadow, cfg->new_theme ? 0 : 200);
  clutter_actor_show (shadow);
 
//  priv->bg = tidy_texture_frame_new (CLUTTER_TEXTURE (iconviewbg),
  //                            BG_PADDING, BG_PADDING, BG_PADDING, BG_PADDING);
  priv->bg = clutter_texture_new_from_pixbuf (iconviewbg_p);
  clutter_container_add_actor (CLUTTER_CONTAINER (view), priv->bg);
  clutter_actor_set_size (priv->bg,
                          view_width+1 + (2*BG_PADDING),
                          view_height+1 + (2*BG_PADDING));
  clutter_actor_set_position (priv->bg, -BG_PADDING, -BG_PADDING);
  clutter_actor_show (priv->bg);
  clutter_actor_set_reactive (priv->bg, TRUE);
  g_signal_connect (priv->bg, "button-release-event",
                    G_CALLBACK (eat_click), NULL);

  priv->grid = tidy_viewport_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (view), priv->grid);
  clutter_actor_set_position (priv->grid, 0, 0);
  clutter_actor_show (priv->grid);

  clutter_actor_set_clip (priv->bg, 0, 0, 0, 0);
  clutter_actor_set_clip (priv->grid, 0, 0, 0, 0);
  clutter_actor_set_clip (shadow, 0, 0, 0, 0);

  priv->cat = cat;

  xoffset =0;
  yoffset =0;

  for (l = launcher_menu_category_get_applications (cat); l; l = l->next)
  {
    LauncherMenuApplication *app = l->data;
    ClutterActor *actor;
    
    actor = launcher_icon_new ();
    
    if (first_run)
      g_object_set_data (G_OBJECT (actor), "app", app);
    else
    {
      launcher_icon_set_application (LAUNCHER_ICON (actor), app);
      launcher_icon_set_position (LAUNCHER_ICON (actor), j);
    }
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->grid), actor);
    clutter_actor_set_size (actor, cfg->icon_width, cfg->icon_height);
    clutter_actor_set_position (actor, xoffset, yoffset);
    clutter_actor_set_reactive (actor, TRUE);
    clutter_actor_show (actor);

    g_signal_connect (actor, "flipped", G_CALLBACK (on_icon_flipped), view);
    g_signal_connect (actor, "enter-event", 
                    G_CALLBACK (on_enter), view);
    g_signal_connect (actor, "leave-event",
                    G_CALLBACK (on_leave), view);

    priv->icons = g_list_append (priv->icons, actor);
        
    if (i == (app_cols -1))
    {
      i = 0;
      xoffset = 0;
      yoffset += cfg->icon_height;
      rows++;
    }
    else
    {
      xoffset += cfg->icon_width;
      i++;
    }
    j++;
  }
  n_icons = j;
  icons = priv->icons;

  clutter_actor_set_height (CLUTTER_ACTOR (priv->grid), 
                            (rows +1) * icon_height);
  /* Add the scroll bar */
  if (g_list_length (launcher_menu_category_get_applications (cat)) 
        > (cfg->iconview_cols * cfg->iconview_rows))
  {
    tidy_scrollable_get_adjustments (TIDY_SCROLLABLE (priv->grid), 
                                     NULL, &adjustment);
    tidy_adjustment_set_values (adjustment, 
                                0, 
                                0, (rows+1) * icon_height,
                                icon_height,
                                3 * icon_height,
                                3 * icon_height);
    priv->adjust = adjustment;

    tex = launcher_util_texture_new_from_file (cfg->handle);
    clutter_container_add_actor (CLUTTER_CONTAINER (view), tex);
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 32, 3, 32, 3);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment); 
    scroll = priv->scroll = tidy_scroll_bar_new_with_handle (adjustment, frame);

    tex = launcher_util_texture_new_from_file (cfg->scrollbar);
    clutter_container_add_actor (CLUTTER_CONTAINER (view), tex);    
    
    frame = tidy_texture_frame_new (CLUTTER_TEXTURE (tex), 30, 5, 30, 5);
    clutter_actor_set_reactive (CLUTTER_ACTOR (frame), TRUE);
    g_signal_connect (frame, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
    clutter_actor_set_reactive (CLUTTER_ACTOR (scroll), TRUE);
    g_signal_connect (scroll, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);
     tidy_frame_set_texture (TIDY_FRAME (scroll), frame);

    clutter_container_add_actor (CLUTTER_CONTAINER (view), scroll);
    clutter_actor_set_size (scroll, view_height, 20);
    clutter_actor_set_anchor_point_from_gravity (scroll, 
                                                 CLUTTER_GRAVITY_CENTER);
    clutter_actor_set_position (scroll, 
                               view_width + (BG_PADDING*1) + 10,
                                view_height/2);
 
    clutter_actor_set_rotation (scroll, CLUTTER_Z_AXIS, 90, 0, 0, 0);
    if (first_run) clutter_actor_set_opacity (scroll, 0);
    clutter_actor_show (scroll);

    clutter_actor_set_reactive (CLUTTER_ACTOR (view), TRUE);
    g_signal_connect (view, "scroll-event",
                      G_CALLBACK (on_scroll_event), adjustment);

    /* The buttons */
    button = clutter_rectangle_new ();
    clutter_container_add_actor (CLUTTER_CONTAINER (view), button);
    clutter_actor_set_size (button, 20, 20);
    clutter_actor_set_position (button, view_width, 0);
    clutter_actor_set_opacity (button, 0);
    clutter_actor_set_reactive (button, TRUE);
    clutter_actor_show (button);
    g_signal_connect (button, "button-release-event", 
                      G_CALLBACK (on_up_pressed), adjustment);

    button = clutter_rectangle_new ();
    clutter_container_add_actor (CLUTTER_CONTAINER (view), button);
    clutter_actor_set_size (button, 20, 20);
    clutter_actor_set_position (button, view_width, view_height-20);
    clutter_actor_set_opacity (button, 0);
    clutter_actor_set_reactive (button, TRUE);
    clutter_actor_show (button);
    g_signal_connect (button, "button-release-event", 
                      G_CALLBACK (on_down_pressed), adjustment);
  }

  /* The lines on the view */
  for (i = 1; i < cfg->n_cats; i++)
  {
    line = clutter_rectangle_new_with_color (&cfg->grid_line_color);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->grid), line);
    clutter_actor_set_size (line, 1, view_height*(rows+3));
    clutter_actor_set_position (line, i * cfg->icon_width, 0);
    clutter_actor_show (line);
  }
  for (i = 1; i < (rows+3); i++)
  {
    line = clutter_rectangle_new_with_color (&cfg->grid_line_color);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->grid), line);
    clutter_actor_set_size (line, view_width, 1);
    clutter_actor_set_position (line, 0, i * cfg->icon_height);
    clutter_actor_show (line);
  }

  /* Add the 'add' button */
  priv->add = launcher_add_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (stage), priv->add);
  clutter_actor_set_size (priv->add, cat_height/4, cat_height/4);
  clutter_actor_set_anchor_point_from_gravity (priv->add, 
                                               CLUTTER_GRAVITY_CENTER);
  clutter_actor_set_position (priv->add, 
                              CSW()-((CSW()/8)/2),
                              ((CSH()/2)-(cfg->iconview_height/2))
                                +cfg->cat_height/2 + PANEL_HEIGHT/2
                                + view_height - (cat_height/8));
  clutter_actor_show (priv->add);
  g_signal_connect (priv->add, "clicked",
                    G_CALLBACK (on_add_clicked), view);
  clutter_focus_manager_insert_group(CLUTTER_FOCUS_MANAGER (manager),
                                     CLUTTER_FOCUS_GROUP (priv->add),
                                     4);

  if (first_run)
  {
    priv->count = 1;
    priv->tag = g_timeout_add (100, (GSourceFunc)foldout, view);
  }
  else
  {
    clutter_actor_remove_clip (priv->bg);
    clutter_actor_set_clip (priv->grid, 
                          0, 
                          0,
                          view_width,
                          view_height);
    clutter_actor_remove_clip (shadow);
  }
} 


/*
 * FOCUS STUFF
 */
static void
launcher_iconview_set_focus (ClutterFocusGroup *group, gboolean has_focus)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;

  if (has_focus)
  {
    priv->focused = priv->icons->data;
    launcher_icon_set_focus (LAUNCHER_ICON (priv->focused), TRUE);
  }
  else
  {
    launcher_icon_set_focus (LAUNCHER_ICON (priv->focused), FALSE);
  }
}

static gboolean
launcher_iconview_direction_event (ClutterFocusGroup     *group,
                                 ClutterFocusDirection  dir)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  LauncherIcon *new;
  GList *children, *c;
  gint current;
  gint next;

  /* Move to the next or previous category, 
   *  if category is first or last, return FALSE
   *  page up and page down takes you to top and bottom
   */
  children = priv->icons;
  current = next = g_list_index (children, priv->focused);

  switch (dir)
  {
    case CLUTTER_DIRECTION_LEFT:
      next -= 1;
      break;
    case CLUTTER_DIRECTION_RIGHT:
      next += 1;
      break;
    case CLUTTER_DIRECTION_UP:
      next -= cfg->iconview_cols;
      break;
    case CLUTTER_DIRECTION_DOWN:
      next += cfg->iconview_cols;
      break;
    case CLUTTER_DIRECTION_PAGE_UP:
      next = 0;
      break;
    case CLUTTER_DIRECTION_PAGE_DOWN:
      next = g_list_length (children) -1;
      break;
    default:
      break;
  }
  if (next < 0) next = 0; 
  next = CLAMP (next, 0, g_list_length (children)-1);
  new = g_list_nth_data (children, next);
  priv->focused = CLUTTER_ACTOR (new);
  launcher_icon_set_focus (new, TRUE);

  for (c = children; c; c= c->next)
  {
    LauncherIcon *icon = c->data;

    if (icon != new)
      launcher_icon_set_focus (icon, FALSE);
  }

  if (TIDY_IS_ADJUSTMENT (priv->adjust))
  {
    gint y, pos, page;

    y = tidy_adjustment_get_value (priv->adjust);
    pos = cfg->icon_height * (next/cfg->iconview_rows);
    page = cfg->iconview_rows * cfg->icon_height;

    if (pos >= (y + page))
    {
      tidy_adjustment_set_value (priv->adjust, pos);
    }
    else if (pos < y)
    {
      tidy_adjustment_set_value (priv->adjust, pos);
    }
  }

  return TRUE;
}

static gboolean
launcher_iconview_action_event (ClutterFocusGroup *group)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW (group)->priv;

  launcher_icon_launch (LAUNCHER_ICON (priv->focused));

  return TRUE;
}

static gboolean
launcher_iconview_key_event (ClutterFocusGroup *group, const gchar *string)
{
  g_debug ("String event: %s", string);

  return TRUE;
}                        

static void
clutter_focus_group_iface_init (ClutterFocusGroupIface *iface)
{
  iface->set_focus       = launcher_iconview_set_focus;
  iface->direction_event = launcher_iconview_direction_event;
  iface->key_event       = launcher_iconview_key_event;
  iface->action_event    = launcher_iconview_action_event;
}

/*
 * /FOCUS STUFF
 */

/* GObject stuff */
static void
launcher_iconview_finalize (GObject *object)
{
  LauncherIconviewPrivate *priv = LAUNCHER_ICONVIEW_GET_PRIVATE (object);

  if (LAUNCHER_IS_ICONVIEW (object) && CLUTTER_IS_ACTOR (priv->scroll))
    clutter_actor_destroy (priv->scroll);

  if (priv->tag)
    g_source_remove (priv->tag);

  g_signal_handlers_disconnect_by_func (clutter_drag_server_get_default (),
                                        on_drag_started,
                                        object);
  g_signal_handlers_disconnect_by_func (clutter_drag_server_get_default (),
                                        on_drag_finished, 
                                        object);


  g_object_unref (iconviewbg);
  g_object_unref (priv->time);
  g_object_unref (priv->temp);
  clutter_actor_destroy (priv->add);

  G_OBJECT_CLASS (launcher_iconview_parent_class)->finalize (object);
}

static void
launcher_iconview_class_init (LauncherIconviewClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_iconview_finalize;

  g_type_class_add_private (obj_class, sizeof (LauncherIconviewPrivate));
}

      
static void
launcher_iconview_init (LauncherIconview *iconview)
{
  LauncherIconviewPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterDragServer *server = clutter_drag_server_get_default ();
  ClutterAlpha *alpha;

  priv = iconview->priv = LAUNCHER_ICONVIEW_GET_PRIVATE (iconview);

  priv->icons = NULL;
  priv->count = priv->tag = 0;

  if (!CLUTTER_IS_TEXTURE (iconviewbg))
    iconviewbg = launcher_util_texture_new_from_file (cfg->iconviewbg);
  else
    g_object_ref (iconviewbg);

  if (!GDK_IS_PIXBUF (iconviewbg_p))
  {
    iconviewbg_p = gdk_pixbuf_new_from_file (cfg->iconviewbg, NULL);
    g_object_ref (iconviewbg_p);
  }

  if (!CLUTTER_IS_TEXTURE (shadowbg))
    shadowbg = launcher_util_texture_new_from_file (PKGDATADIR"/shadow.svg");
  else
    g_object_ref (shadowbg);


  priv->time = launcher_util_timeline_new_slow ();
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);

  priv->reorder_time = launcher_util_timeline_new_slow ();
  alpha = clutter_alpha_new_full (priv->reorder_time, 
                                  clutter_ramp_inc_func,
                                  NULL, NULL);
  priv->reorder_behave = launcher_behave_new (alpha, 
                                      (LauncherBehaveAlphaFunc)reorder_func,
                                      (gpointer)iconview);

  g_signal_connect (server, "drag-started", 
                    G_CALLBACK (on_drag_started), iconview);
  g_signal_connect (server, "drag-finished", 
                    G_CALLBACK (on_drag_finished), iconview);
}

ClutterActor *
launcher_iconview_new (void)

{
  ClutterActor *iconview = NULL;

  iconview = g_object_new (LAUNCHER_TYPE_ICONVIEW, 
                           "sync-adjustments", FALSE,  
                           NULL);

  return iconview;
}
