/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>

#include <libgnomeui/libgnomeui.h>

#include "launcher-shortcut-editor.h"

#include "launcher-app.h"
#include "launcher-application-chooser.h"
#include "launcher-defines.h"
#include "launcher-menu.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherShortcutEditor, launcher_shortcut_editor, GTK_TYPE_DIALOG);

#define LAUNCHER_SHORTCUT_EDITOR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_SHORTCUT_EDITOR, \
  LauncherShortcutEditorPrivate))

#define WEB_ICON_NAME "applications-internet"
#define FILE_ICON_NAME "gnome-fs-regular"
#define FOLDER_ICON_NAME "gnome-fs-directory"
#define APP_ICON_NAME "gnome-panel-launcher"

struct _LauncherShortcutEditorPrivate
{
  LauncherMenuCategory *cat;

  GtkWidget *combo;
  GtkWidget *name_entry;
  GtkWidget *loc_entry;
  GtkWidget *icon;
  GtkWidget *browse;

  gboolean icon_change_lock;
};

enum
{
	PROP_0,
  PROP_TYPE,
	PROP_NAME,
  PROP_EXEC,
  PROP_ICON
};

static const gchar *icon_names[] =  
{ 
  WEB_ICON_NAME,
  FILE_ICON_NAME,
  FOLDER_ICON_NAME, 
  APP_ICON_NAME 
};

  
static void
on_entry_activate (GtkEntry *entry, GtkDialog *self)
{
  gtk_dialog_response (GTK_DIALOG (self), GTK_RESPONSE_YES);
}

static void
set_icon_path (GnomeIconEntry *entry, const gchar *icon_name)
{
  GtkIconTheme *theme = gtk_icon_theme_get_default ();
  GtkIconInfo *info = NULL;

  info = gtk_icon_theme_lookup_icon (theme, icon_name, 
                                     48, 
                                     GTK_ICON_LOOKUP_FORCE_SVG);

  if (info)
  {
    gnome_icon_entry_set_filename (entry, gtk_icon_info_get_filename (info));
    gtk_icon_info_free (info);
  }
  else
  {
    gnome_icon_entry_set_filename (entry, icon_name);
  }
  g_object_set (entry, "pixmap-subdir", "/usr/share/pixmaps", NULL);
}

static void
on_combo_changed (GtkComboBox *combo, LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv;
  gint active;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));
  priv = self->priv;

  active = gtk_combo_box_get_active (combo);

  if (!priv->icon_change_lock)
    set_icon_path (GNOME_ICON_ENTRY (priv->icon), icon_names[active]);

  if (active == SHORTCUT_TYPE_WEB)
    gtk_widget_set_sensitive (priv->browse, FALSE);
  else
    gtk_widget_set_sensitive (priv->browse, TRUE);
}

/* Taken from nautilus::panel-util.c, with some modifications */
static gchar *
get_icon_name (GIcon *icon)
{
  const gchar * const *names;
  GtkIconTheme *icon_theme;
  int i;

  if (!G_IS_THEMED_ICON (icon))
    return g_strdup ("gnome-fs-folder");

  names = g_themed_icon_get_names (G_THEMED_ICON (icon));
  icon_theme = gtk_icon_theme_get_default ();

  for (i = 0; names[i] != NULL; i++)
  {
    if (gtk_icon_theme_has_icon (icon_theme, names[i]))
          return g_strdup (names[i]);
  }

  return NULL;
}

static gchar *
_get_icon (const gchar *filename)
{
  gchar *thumb;
  gchar *ret = NULL;

  thumb = gnome_thumbnail_path_for_uri (filename, GNOME_THUMBNAIL_SIZE_NORMAL);
  
  if ((g_file_test (thumb, G_FILE_TEST_EXISTS)))
    ret = g_strdup (thumb);

  if (!ret)
  {
    GFile *file;
    GFileInfo *info;
    GIcon *icon;

    file = g_file_new_for_path (filename);
    info = g_file_query_info (file, G_FILE_ATTRIBUTE_STANDARD_ICON,
                              0, NULL, NULL);
    icon = (GIcon*)g_file_info_get_attribute_object (info,  
                                              G_FILE_ATTRIBUTE_STANDARD_ICON);

    ret = get_icon_name (icon);

    g_object_unref (file);
    g_object_unref (info);
  }

  if (!ret)
    ret = g_strdup ("gnome-fs-folder");

  return ret;
}

static void
on_file_browse_response (GtkWidget              *dialog,
                         gint                    response,
                         LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv;
  gchar *filename, *icon;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));
  priv = self->priv;

  if (response == GTK_RESPONSE_YES)
  {
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    icon = _get_icon (filename);
    
    gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), filename);
    set_icon_path (GNOME_ICON_ENTRY (priv->icon), icon);
    
    g_free (filename);
    g_free (icon);
  }

  gtk_widget_set_sensitive (GTK_WIDGET (self), TRUE);
  gtk_widget_destroy (dialog);
}

static void
on_app_browse_response (GtkWidget              *dialog,
                        gint                    response,
                        LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));
  priv = self->priv;

  if (response == GTK_RESPONSE_YES)
  {
    gchar *name = NULL, *exec = NULL, *icon = NULL;

    if (launcher_application_chooser_get_selected 
          (LAUNCHER_APPLICATION_CHOOSER (dialog), 
           &name, &exec, &icon))
    {
      gtk_entry_set_text (GTK_ENTRY (priv->name_entry), name);
      gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), exec);
      set_icon_path (GNOME_ICON_ENTRY (priv->icon), icon);

      g_free (name);
      g_free (exec);
      g_free (icon);
    }
  }

  gtk_widget_set_sensitive (GTK_WIDGET (self), TRUE);
  gtk_widget_destroy (dialog);
}


static void
on_browse_clicked (GtkWidget *button, LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv;
  gint active;
  GtkWidget *dialog;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));
  priv = self->priv;

  active = gtk_combo_box_get_active (GTK_COMBO_BOX (priv->combo));

  if (active == SHORTCUT_TYPE_FILE)
  {

    dialog = gtk_file_chooser_dialog_new (_("Choose location"),
                                          GTK_WINDOW (self),
                                          GTK_FILE_CHOOSER_ACTION_OPEN,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          GTK_STOCK_OPEN, GTK_RESPONSE_YES,
                                        NULL);
    gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (dialog), FALSE);
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), 
                                         g_get_home_dir ());    
    g_signal_connect (dialog, "response", 
                      G_CALLBACK (on_file_browse_response), self);

  }
  else if (active == SHORTCUT_TYPE_FOLDER)
  {

    dialog = gtk_file_chooser_dialog_new (_("Choose location"),
                                          GTK_WINDOW (self),
                                         GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          GTK_STOCK_OPEN, GTK_RESPONSE_YES,
                                        NULL);
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), 
                                         g_get_home_dir ());
    gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (dialog), FALSE);
    g_signal_connect (dialog, "response", 
                      G_CALLBACK (on_file_browse_response), self);
  }
  else
  {
    dialog = launcher_application_chooser_new ();
    gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OPEN, GTK_RESPONSE_YES,
                            NULL);
 
    g_signal_connect (dialog, "response", 
                      G_CALLBACK (on_app_browse_response), self);
  }

  launcher_util_present_window (GTK_WINDOW (dialog));
  gtk_widget_set_sensitive (GTK_WIDGET (self), FALSE);
}

void
launcher_shortcut_editor_set_name (LauncherShortcutEditor *self, 
                                   const gchar            *name)
{
  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));

  gtk_entry_set_text (GTK_ENTRY (self->priv->name_entry), name);
}

void
launcher_shortcut_editor_set_exec (LauncherShortcutEditor *self, 
                                   const gchar            *exec)
{
  LauncherShortcutEditorPrivate *priv;
  gint type = SHORTCUT_TYPE_APP;
  static gint open_len = 0;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));
  priv = self->priv;

  if (!open_len)
    open_len = strlen (XDG_OPEN) + 1;

  if (!exec)
    return;

  /* Yes this isn't the best way, but it's all I have time for */
  if (strstr (exec, XDG_OPEN))
  {
    if (exec && strstr (exec, "http"))
      type = SHORTCUT_TYPE_WEB;
    else if (exec && strstr (exec, "www"))
      type = SHORTCUT_TYPE_WEB;
    else if (exec && strstr (exec, "/"))
    {
      gchar **tokens = g_strsplit (exec, " ", 2);
      
      if ((g_file_test (tokens[1], G_FILE_TEST_IS_DIR)))
        type = SHORTCUT_TYPE_FOLDER;
      else
        type = SHORTCUT_TYPE_FILE;

      g_strfreev (tokens);
    }
  }

  gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo), type);

  if (exec && strstr (exec, XDG_OPEN))
  {
    gchar *temp = g_strdup (&exec[open_len]);
    gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), temp);
    g_free (temp);
  }
  else
    gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), exec);
}

gchar *
launcher_shortcut_editor_get_exec (LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv = self->priv;
  gchar *exec = NULL;
  const gchar *text;

  text = gtk_entry_get_text (GTK_ENTRY (priv->loc_entry));

  switch (gtk_combo_box_get_active (GTK_COMBO_BOX (priv->combo)))
  {
    case SHORTCUT_TYPE_APP:
      exec = g_strdup (gtk_entry_get_text (GTK_ENTRY (priv->loc_entry)));
      break;
    default:
      if (text && text[0] == '"')
        exec = g_strdup_printf ("%s %s", XDG_OPEN, text);
      else
        exec = g_strdup_printf ("%s \"%s\"", XDG_OPEN,
                              gtk_entry_get_text (GTK_ENTRY (priv->loc_entry)));
  }

  return exec;
}

void
launcher_shortcut_editor_set_icon (LauncherShortcutEditor *self, 
                                   const gchar            *icon_name)
{
  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (self));

  set_icon_path (GNOME_ICON_ENTRY (self->priv->icon), icon_name);
  self->priv->icon_change_lock = TRUE;
}

/* GObject stuff */
static void
launcher_shortcut_editor_finalize (GObject *shortcut_editor)
{
  LauncherShortcutEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (shortcut_editor));
  priv = LAUNCHER_SHORTCUT_EDITOR_GET_PRIVATE (shortcut_editor);

  G_OBJECT_CLASS (launcher_shortcut_editor_parent_class)->finalize (shortcut_editor);
}

static void
set_property (GObject      *object, 
              guint         prop_id,
              const GValue *value, 
              GParamSpec   *pspec)
{
  LauncherShortcutEditor *self = LAUNCHER_SHORTCUT_EDITOR (object);
  LauncherShortcutEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (object));
  priv = LAUNCHER_SHORTCUT_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo),  
                                g_value_get_int (value));
      break;
    case PROP_NAME:
      launcher_shortcut_editor_set_name (self, g_value_get_string (value));
      break;
    case PROP_EXEC:
      launcher_shortcut_editor_set_exec (self, g_value_get_string (value));
      break;
    case PROP_ICON:
      launcher_shortcut_editor_set_icon (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
get_property (GObject      *object, 
              guint         prop_id,
              GValue       *value, 
              GParamSpec   *pspec)
{
  LauncherShortcutEditorPrivate *priv;
  gchar *exec;

  g_return_if_fail (LAUNCHER_IS_SHORTCUT_EDITOR (object));
  priv = LAUNCHER_SHORTCUT_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      g_value_set_int (value,
                       gtk_combo_box_get_active (GTK_COMBO_BOX (priv->combo)));

      break;
    case PROP_NAME:
      g_value_set_string (value, 
                          gtk_entry_get_text (GTK_ENTRY (priv->name_entry)));
      break;
    case PROP_EXEC:
      exec = launcher_shortcut_editor_get_exec (
                                            LAUNCHER_SHORTCUT_EDITOR(object));
      g_value_set_string (value, exec);
      g_free (exec);
      break;
    case PROP_ICON:
      g_value_set_string (value, 
                gnome_icon_entry_get_filename (GNOME_ICON_ENTRY (priv->icon)));
       break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
launcher_shortcut_editor_class_init (LauncherShortcutEditorClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_shortcut_editor_finalize;
  obj_class->get_property = get_property;
  obj_class->set_property = set_property;


  /* Class properties */
  g_object_class_install_property (obj_class,
                                   PROP_TYPE,
                                   g_param_spec_int ("shortcut-type",
                                   "Shortcut Type",
                                   "Type of shortcut",
                                   0, 4, 0,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_NAME,
                                   g_param_spec_string ("shortcut-name",
                                   "Shortcut Name",
                                   "Name of shortcut",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_EXEC,
                                   g_param_spec_string ("shortcut-exec",
                                   "Shortcut Exec",
                                   "Exec of shortcut",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_ICON,
                                   g_param_spec_string ("shortcut-icon",
                                   "Shortcut Icon",
                                   "Icon of shortcut",
                                   NULL,
                                   G_PARAM_READWRITE));


  g_type_class_add_private (obj_class, sizeof (LauncherShortcutEditorPrivate));
}

      
static void
launcher_shortcut_editor_init (LauncherShortcutEditor *self)
{
  LauncherShortcutEditorPrivate *priv;
  GtkWidget *window = GTK_WIDGET (self);
  GtkWidget *align, *vbox, *hbox, *hbox2;
  GtkWidget *label, *combo, *entry, *button, *icon;
  GtkSizeGroup *group;
  
  priv = self->priv = LAUNCHER_SHORTCUT_EDITOR_GET_PRIVATE (self);

  launcher_app_lock (launcher_app_get_default (), TRUE);

  priv->icon_change_lock = FALSE;

  group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 0, 12, 12);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (window)->vbox), align);
  g_object_set (GTK_DIALOG(window)->vbox,  "spacing", 8, NULL);  
  
  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_add (GTK_CONTAINER (align), vbox);
  
  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Type:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  combo = priv->combo = gtk_combo_box_new_text ();
  gtk_widget_set_size_request (combo, 300, -1);
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("Web Site"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("File"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("Folder"));
  gtk_combo_box_append_text (GTK_COMBO_BOX (combo), _("Application"));
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
  gtk_box_pack_start (GTK_BOX (hbox), combo, 0, TRUE, TRUE);
  g_signal_connect (combo, "changed",
                    G_CALLBACK (on_combo_changed), self);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);
  
  entry = priv->name_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);
  g_signal_connect (entry, "activate", G_CALLBACK (on_entry_activate), self);
  gtk_widget_grab_focus (entry);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Location:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  hbox2 = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, 0, FALSE, FALSE);
 
  entry = priv->loc_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox2), entry, 0, TRUE, TRUE);
  g_signal_connect (entry, "activate", G_CALLBACK (on_entry_activate), self); 

  button = priv->browse = gtk_button_new_with_label (_("browse..."));
  gtk_widget_set_sensitive (button, FALSE);
  gtk_box_pack_start (GTK_BOX (hbox2), button, 0, FALSE, FALSE);
  g_signal_connect (button, "clicked", G_CALLBACK (on_browse_clicked), self);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Icon:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  icon = priv->icon = gnome_icon_entry_new ("1", _("Choose shortcut icon"));
  set_icon_path (GNOME_ICON_ENTRY (icon), icon_names[0]);
  gtk_box_pack_start (GTK_BOX (hbox), icon, 0, TRUE, TRUE);
  
  gtk_widget_show_all (align);
}

GtkWidget *
launcher_shortcut_editor_new (void)

{
  GtkWidget *shortcut_editor = NULL;

  shortcut_editor = g_object_new (LAUNCHER_TYPE_SHORTCUT_EDITOR, 
                                  "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                                  NULL);

  return shortcut_editor;
}
