/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
#include <string.h>
#include <stdlib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gio/gio.h>

#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-icon.h"
#include "launcher-util.h"

#include <libgnome/gnome-desktop-item.h>

/* Convert command line to argv array, stripping % conversions on the way */
#define MAX_ARGS 255

/*
 * Convert a desktop file exec string to an array of strings suitable for
 * passing to gdk_spawn_on_screen. Code from libtaku (matchbox-desktop-2).
 */
gchar**
launcher_util_exec_to_argv (const gchar *exec)
{
  const char *p;
  char *buf, *bufp, **argv;
  int nargs;
  gboolean escape, single_quote, double_quote;
  
  argv = g_new (char *, MAX_ARGS + 1);
  buf = g_alloca (strlen (exec) + 1);
  bufp = buf;
  nargs = 0;
  escape = single_quote = double_quote = FALSE;
  
  for (p = exec; *p; p++) {
    if (escape) {
      *bufp++ = *p;
      
      escape = FALSE;
    } else {
      switch (*p) {
      case '\\':
        escape = TRUE;

        break;
      case '%':
        /* Strip '%' conversions */
        if (p[1] && p[1] == '%')
          *bufp++ = *p;
        
        p++;

        break;
      case '\'':
        if (double_quote)
          *bufp++ = *p;
        else
          single_quote = !single_quote;
        
        break;
      case '\"':
        if (single_quote)
          *bufp++ = *p;
        else
          double_quote = !double_quote;
        
        break;
      case ' ':
        if (single_quote || double_quote)
          *bufp++ = *p;
        else {
          *bufp = 0;
          
          if (nargs < MAX_ARGS)
            argv[nargs++] = g_strdup (buf);
          
          bufp = buf;
        }
        
        break;
      default:
        *bufp++ = *p;
        break;
      }
    }
  }
  
  if (bufp != buf) {
    *bufp = 0;
    
    if (nargs < MAX_ARGS)
      argv[nargs++] = g_strdup (buf);
  }
  
  argv[nargs] = NULL;
  
  return argv;
}


ClutterActor * 
launcher_util_texture_new_from_file (const gchar *filename)
{
  ClutterActor *texture = NULL;
  GdkPixbuf    *pixbuf;
  
  pixbuf = gdk_pixbuf_new_from_file (filename, NULL);

  if (!GDK_IS_PIXBUF (pixbuf))
  {
    g_warning ("Unable to load %s\n", filename);
  }
  else
  {
    texture = clutter_texture_new_from_pixbuf (pixbuf);
  }

  if (G_IS_OBJECT (pixbuf)) 
    g_object_unref (pixbuf);

  return texture;
}

/* From matchbox-desktop */
static char *
strip_extension (const char *file)
{
        char *stripped, *p;

        stripped = g_strdup (file);

        p = strrchr (stripped, '.');
        if (p &&
            (!strcmp (p, ".png") ||
             !strcmp (p, ".svg") ||
             !strcmp (p, ".xpm")))
	        *p = 0;

        return stripped;
}

static GdkPixbuf *
constrain_size (GdkPixbuf *pixbuf)
{
#define MAX_WIDTH ((67))
  static LauncherConfig *cfg = NULL;
  GdkPixbuf *temp = NULL;
  gint width;
  
  if (cfg == NULL)
    cfg = launcher_config_get_default ();

  width = gdk_pixbuf_get_width (pixbuf);

  if (width > MAX_WIDTH)
  {
    gint height;

    height = gdk_pixbuf_get_height (pixbuf);
    
    temp = pixbuf;

    pixbuf = gdk_pixbuf_scale_simple (temp, 
                                      MAX_WIDTH,
    (gint)(((gfloat)height/width) * (MAX_WIDTH)),
                                      GDK_INTERP_BILINEAR);
    g_object_unref (temp);
  }
  return pixbuf;
}

/* Gets the pixbuf from a desktop file's icon name. Based on the same function
 * from matchbox-desktop
 */
GdkPixbuf *
get_icon (const gchar *name, guint size)
{
  static GtkIconTheme *theme = NULL;
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;
  gchar *stripped = NULL;

  gint width, height;

  if (theme == NULL)
    theme = gtk_icon_theme_get_default ();

  if (name == NULL)
  {
    pixbuf = gtk_icon_theme_load_icon (theme, "application-x-executable",
                                       size, 0, NULL);
    return pixbuf;
  }

  if (g_path_is_absolute (name))
  {
    if (g_file_test (name, G_FILE_TEST_EXISTS))
    {
      pixbuf = gdk_pixbuf_new_from_file_at_scale (name, -1, size, 
                                                  TRUE, &error);
      if (error)
      {
        /*g_warning ("Error loading icon: %s\n", error->message);*/
        g_error_free (error);
        error = NULL;
     }
      return constrain_size (pixbuf);
    } 
  }

  stripped = strip_extension (name);

  {  
    GtkIconInfo *info;
    const gchar *filename;
    
    info = gtk_icon_theme_lookup_icon (theme, stripped, size, 0);

    if (info)
    {
      filename = gtk_icon_info_get_filename (info);
      
      if (filename)
      {
        pixbuf = gdk_pixbuf_new_from_file_at_scale (filename, -1, size, 
                                                  TRUE, &error);
        if (error)
        {
          /*g_warning ("Error loading icon: %s\n", error->message);*/
          g_error_free (error);
          error = NULL;
        }
      }
      gtk_icon_info_free (info);
    }
  }
  
  if (!GDK_IS_PIXBUF (pixbuf))
  {
    pixbuf = gtk_icon_theme_load_icon (theme,
                                       stripped,
                                       size,
                                       0, &error);
    if (error)
    {   
      /*g_warning ("Error loading icon: %s\n", error->message);*/
      g_error_free (error);
      error = NULL;
    }
  }
  /* Always try and send back something */
  if (!GDK_IS_PIXBUF (pixbuf))
    pixbuf = gtk_icon_theme_load_icon (theme, "application-x-executable",
                                       size, 
                                       0, NULL);

  if (GDK_IS_PIXBUF (pixbuf))
  {  
    width = gdk_pixbuf_get_width (pixbuf);
    height = gdk_pixbuf_get_height (pixbuf);

    if (height != size && 0)
    {
      GdkPixbuf *temp = pixbuf;
      pixbuf = gdk_pixbuf_scale_simple (temp, 
                                      (gint)((float)width/height)*size,
                                      size,
                                      GDK_INTERP_HYPER);
      g_object_unref (temp);
    }
  }
  g_free (stripped);

  return pixbuf;
}



ClutterActor * 
launcher_util_texture_new_from_named_icon (const gchar *name)
{
  GdkPixbuf *pixbuf;
  ClutterActor *texture;

  pixbuf = get_icon (name, 32);
  
  texture = clutter_texture_new_from_pixbuf (pixbuf);

  g_object_unref (pixbuf);

  return texture;
}

void
_normalize (char *str)
{
	size_t len;
	int i;

	len = strlen(str);

	for (i = 0; i < len; i++) 
  {
		switch (str[i]) 
    {
			case ' ':
      case '/':
				str[i] = '-';
        break;
			default:
				break;
		}
	}

}

void           
launcher_util_create_favorite (LauncherMenuApplication *app)
{
  GError *error = NULL;
  GnomeDesktopItem *item = NULL;
  GnomeDesktopItem *fav = NULL;
  const gchar *item_file;
  gchar *appdir, *filename, *favfile;

  g_return_if_fail (app);

  item_file = launcher_menu_application_get_desktop_filename (app);
  item = gnome_desktop_item_new_from_file (item_file,
                                         GNOME_DESKTOP_ITEM_LOAD_ONLY_IF_EXISTS,
                                           NULL);
  if (item == NULL)
    return;

  fav = gnome_desktop_item_copy (item);
  gnome_desktop_item_set_string (fav, 
                                 GNOME_DESKTOP_ITEM_CATEGORIES,
                                 "Favorites");

  appdir = g_strdup_printf ("file://%s/.local/share/applications", 
                            g_get_home_dir ());
  filename = g_strdup_printf ("%s-favorite-%ld.desktop",
                              launcher_menu_application_get_name (app),
                              time (NULL));
  _normalize (filename);
  favfile = g_build_filename (appdir, filename, NULL);
                               
  gnome_desktop_item_save (fav, favfile, TRUE, &error);

  if (error)
  {
    g_warning ("Unable to create favorite: %s", error->message);
    g_error_free (error);
  }

  g_free (appdir);
  g_free (filename);
  g_free (favfile);

  gnome_desktop_item_unref (item);
  gnome_desktop_item_unref (fav);
}

void
launcher_util_remove_favorite (LauncherMenuApplication *app)
{
  GnomeDesktopItem *item;
  const gchar *path;
  gchar *filename;
  gchar *basename;
  GError *error = NULL;

  g_return_if_fail (app);

  path = launcher_menu_application_get_desktop_filename (app);

  if ((g_remove (path) == 0))
  {
    return;
  }

  item = gnome_desktop_item_new_from_file (path,
                                        GNOME_DESKTOP_ITEM_LOAD_ONLY_IF_EXISTS, 
                                        NULL);
  if (!item)
    return;

  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_NO_DISPLAY, 
                                 "true");
  basename = g_path_get_basename (path);
  filename = g_build_filename ("file://",
                               g_get_home_dir (), 
                               ".local/share/applications",
                               basename,
                               NULL);
  gnome_desktop_item_save (item, filename, TRUE, &error);

  if (error)
  {
    g_warning ("Unable to remove favorite: %s", error->message);
    g_error_free (error);
  }
  g_free (basename);
  g_free (filename);

  gnome_desktop_item_unref (item);

}

void  
launcher_util_new_shortcut (const gchar *uri,
                            const gchar *name,
                            const gchar *category,
                            const gchar *exec,
                            const gchar *icon)
{
  GError *error = NULL;
  GnomeDesktopItem *item = NULL;

  item = gnome_desktop_item_new ();

  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_NAME,
                                 name); 
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_CATEGORIES,
                                 category);
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_EXEC,
                                 exec);
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_ICON,
                                 icon);
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_TYPE,
                                 "Application");
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_VERSION,
                                 "1.0");
                        
  g_debug ("Creating new favourite: %s", uri);

  gnome_desktop_item_save (item, uri, TRUE, &error);
  if (error)
  {
    g_warning ("Unable to create favorite: %s", error->message);
    g_error_free (error);
  }

  gnome_desktop_item_unref (item);
}

void   
launcher_util_edit_shortcut (const gchar *uri,
                             const gchar *category,
                             const gchar *name,
                             const gchar *exec,
                             const gchar *icon)
{
  GError *error = NULL;
  GnomeDesktopItem *item = NULL;

  item = gnome_desktop_item_new_from_uri (uri, 
                                          0,
                                          &error);
  if (error)
  {
    g_warning ("Unable to open favorite: %s", error->message);
    g_error_free (error);
    return;
  }

  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_NAME,
                                 name); 
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_CATEGORIES,
                                 category);
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_EXEC,
                                 exec);
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_ICON,
                                 icon);
  
  gnome_desktop_item_save (item, uri, TRUE, &error);
  if (error)
  {
    g_warning ("Unable to edit favorite: %s", error->message);
    g_error_free (error);
  }

  gnome_desktop_item_unref (item);
}

static gboolean
write_menu_file (const gchar *uri, const gchar *id, const gchar *name)
{
  static gchar *merge_menu[] = {
  "<!DOCTYPE Menu PUBLIC \"-//freedesktop//DTD Menu 1.0//EN\""
  "\"http://www.freedesktop.org/standards/menu-spec/menu-1.0.dtd\">"
  "<Menu>"
  "<Name>Applications</Name>"
  "  <Menu>"
  "    <Name>%s</Name>"
  "    <Directory>%s.directory</Directory>"
  "    <Include>"
  "      <And>"
  "        <Category>%s</Category>"
  "      </And>"
  "    </Include>"
  "  </Menu>"
  "</Menu>"};
  GFile *file;
  GFileOutputStream *stream;
  GError *error = NULL;
  gchar *buffer = NULL;
  gsize written = 0;
  gchar *leaf, *path;

  leaf = g_strdup_printf ("%s.menu", id);
  _normalize (leaf);
  path = g_build_filename (g_get_home_dir (), 
                           MENU_FILE_PATH,
                           leaf, 
                           NULL);
  file = g_file_new_for_path (path);

  stream = g_file_create (file, G_FILE_CREATE_NONE, NULL, &error);
  if (error)
  {
    g_warning ("Error creating .menu file: %s", error->message);
    g_error_free (error);
    g_free (path);
    g_free (leaf);
    g_object_unref (file);
    return FALSE;
  }

  buffer = g_strdup_printf (*merge_menu, name, id, id); 
                            

  g_output_stream_write_all (G_OUTPUT_STREAM (stream),
                             buffer,
                             strlen (buffer),
                             &written,
                             NULL, 
                             &error);
  if (error)
  {
    g_warning ("Error writing data: %s", error->message);
    g_error_free (error);
    g_free (path);
    g_free (leaf);
    g_object_unref (file);
    g_object_unref (stream);
    return FALSE;
  }
                           
  g_assert (written == strlen (buffer));

  g_output_stream_close (G_OUTPUT_STREAM (stream), NULL, &error);
  if (error)
  {
    g_warning ("Error saving file: %s", error->message);
    g_error_free (error);
    g_free (path);
    g_free (leaf);
    g_object_unref (file);
    g_object_unref (stream);
    return FALSE;
  }

  g_object_unref (stream);
  g_object_unref (file);
  g_free (buffer);
  g_free (path);
  g_free (leaf);


  return TRUE;

}

void     
launcher_util_new_category (const gchar *uri,
                            const gchar *id,
                            const gchar *name,
                            const gchar *icon)
{
  GError *error = NULL;
  GnomeDesktopItem *item = NULL;

  item = gnome_desktop_item_new ();

  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_NAME,
                                 name); 
 
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_ICON,
                                 icon);
  gnome_desktop_item_set_string (item,
                                GNOME_DESKTOP_ITEM_COMMENT,
                                " ");
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_TYPE,
                                 "Directory");
                        
  g_debug ("Creating new category: %s", uri);

  gnome_desktop_item_save (item, uri, TRUE, &error);
  if (error)
  {
    g_warning ("Unable to create category: %s", error->message);
    g_error_free (error);
  }

  write_menu_file (uri, id, name);

  gnome_desktop_item_unref (item);
}

void 
launcher_util_edit_category (const gchar *uri,
                             const gchar *name,
                             const gchar *icon)
{
  GError *error = NULL;
  GnomeDesktopItem *item = NULL;

  item = gnome_desktop_item_new_from_uri (uri, 
                                        GNOME_DESKTOP_ITEM_LOAD_ONLY_IF_EXISTS,
                                          &error);
  if (error)
  {
    g_warning ("Unable to open category: %s", error->message);
    g_error_free (error);
    return;
  }

  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_NAME,
                                 name); 
  gnome_desktop_item_set_string (item, 
                                 GNOME_DESKTOP_ITEM_ICON,
                                 icon);
  
  gnome_desktop_item_save (item, uri, TRUE, &error);
  if (error)
  {
    g_warning ("Unable to edit category: %s", error->message);
    g_error_free (error);
  }

  gnome_desktop_item_unref (item);
}

void    
launcher_util_remove_category (LauncherMenuCategory *category)
{
  GFile *file;
  gchar *menufile, *leaf;

  leaf = g_strdup_printf ("%s.menu", category->id);
  menufile = g_build_filename (g_get_home_dir (),
                               MENU_FILE_PATH,
                               leaf,
                               NULL);

  file = g_file_new_for_path (menufile);
  g_file_delete (file, NULL, NULL);
  g_object_unref (file);

  file = g_file_new_for_path (category->path);
  g_file_delete (file, NULL, NULL);
  g_object_unref (file);

}

/*
GdkPixbuf * surface_2_pixbuf( GdkPixbuf * pixbuf, cairo_surface_t * surface)

original code from abiword (http://www.abisource.com/)  go-image.c
Function name:  static void pixbuf_to_cairo (GOImage *image);
Copyright (C) 2004, 2005 Jody Goldberg (jody@gnome.org)

modified by Rodney Cryderman (rcryderman@gmail.com).

Send it a allocated pixbuff and cairo image surface.  the heights and width 
must match.  Both must be ARGB.
will copy from the surface to the pixbuf.

*/

static GdkPixbuf * 
surface_2_pixbuf( GdkPixbuf * pixbuf, cairo_surface_t * surface)
{
	guint i, j;
	
	guint src_rowstride,dst_rowstride;
	guint src_height, src_width, dst_height, dst_width;
	
	unsigned char *src, *dst;
	guint t;

#define MULT(d,c,a,t) G_STMT_START { t = (a)? c * 255 / a: 0; d = t;} G_STMT_END
	dst = gdk_pixbuf_get_pixels (pixbuf);
	dst_rowstride = gdk_pixbuf_get_rowstride (pixbuf);
	dst_width=gdk_pixbuf_get_width (pixbuf);
	dst_height= gdk_pixbuf_get_height (pixbuf);
	src_width=cairo_image_surface_get_width (surface);
	src_height=cairo_image_surface_get_height (surface);
	src_rowstride=cairo_image_surface_get_stride (surface);
	src = cairo_image_surface_get_data (surface);

  g_return_val_if_fail( src_width == dst_width, NULL );
  g_return_val_if_fail( src_height == dst_height, NULL  );   
  g_return_val_if_fail( 
                cairo_image_surface_get_format(surface) == CAIRO_FORMAT_ARGB32,
                NULL);

	for (i = 0; i < dst_height; i++) 
	{
		for (j = 0; j < dst_width; j++) {
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
			MULT(dst[0], src[2], src[3], t);
			MULT(dst[1], src[1], src[3], t);
			MULT(dst[2], src[0], src[3], t);
			dst[3] = src[3];
#else	  
			MULT(dst[3], src[2], src[3], t);
			MULT(dst[2], src[1], src[3], t);
			MULT(dst[1], src[0], src[3], t);
			dst[0] = src[3];
#endif
			src += 4;
			dst += 4;
		}
		dst += dst_rowstride - dst_width * 4;
		src += src_rowstride - src_width * 4;
			
	}
#undef MULT
    return pixbuf;
}


GdkPixbuf * 
launcher_util_get_pixbuf_from_surface (cairo_surface_t *surface)
{
  GdkPixbuf *pixbuf;
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB,TRUE, 
                           8, 
                           cairo_image_surface_get_width (surface),
                           cairo_image_surface_get_height (surface));
  g_return_val_if_fail (pixbuf != NULL, NULL);

  return surface_2_pixbuf (pixbuf, surface);
}


/*
 * Taken from libtomboy, (C) 2008 Novell, LGPL v2 or later
 */

static void
tomboy_window_override_user_time (GtkWindow *window)
{
	guint32 ev_time = gtk_get_current_event_time();

	if (ev_time == 0) {
		/* 
		 * FIXME: Global keypresses use an event filter on the root
		 * window, which processes events before GDK sees them.
		 */
		//ev_time = tomboy_keybinder_get_current_event_time ();
    ev_time = CLUTTER_CURRENT_TIME;
	}
	if (ev_time == 0) {
		gint ev_mask = gtk_widget_get_events (GTK_WIDGET(window));
		if (!(ev_mask & GDK_PROPERTY_CHANGE_MASK)) {
			gtk_widget_add_events (GTK_WIDGET (window),
					       GDK_PROPERTY_CHANGE_MASK);
		}

		/* 
		 * NOTE: Last resort for D-BUS or other non-interactive
		 *       openings.  Causes roundtrip to server.  Lame. 
		 */
		ev_time = gdk_x11_get_server_time (GTK_WIDGET(window)->window);
	}

	gdk_x11_window_set_user_time (GTK_WIDGET(window)->window, ev_time);
}



void 
launcher_util_present_window (GtkWindow *window)
{
	if (!GTK_WIDGET_REALIZED (window))
		gtk_widget_realize (GTK_WIDGET (window));

	tomboy_window_override_user_time (window);

	gtk_window_present (window);
}

static guint fps = 0;
static guint frames = 0;

static void
load_values ()
{
  LauncherConfig *cfg = launcher_config_get_default ();
  const gchar *fpsenv = g_getenv ("LAUNCHER_FPS");
  const gchar *framesenv = g_getenv ("LAUNCHER_FRAMES");
  if (fpsenv)
      fps = atoi (fpsenv);
  else
    fps = 75;

  if (framesenv)
    frames = atoi (framesenv);
  else
    frames = 24;

  /* Load up different values for Poulsbo */
  if (!framesenv && !fpsenv && cfg->is_poulsbo)
  {
    frames = 15;
    fps = 40;
  }

  g_debug ("FRAMES=%d FPS=%d\n", frames, fps);
}

ClutterTimeline *
launcher_util_timeline_new_slow ()
{
  if (!fps)
    load_values ();

  return clutter_timeline_new (frames + (100 * (fps/1000)), fps);
}

ClutterTimeline * 
launcher_util_timeline_new_medium ()
{
  if (!fps)
    load_values ();

  return clutter_timeline_new (frames, fps);
}

ClutterTimeline * 
launcher_util_timeline_new_fast ()
{
  if (!fps)
    load_values ();

  return clutter_timeline_new (frames - (100 * (fps/1000)), fps);
}

static ClutterActor *eat = NULL;
static guint eat_tag = 0;

static gboolean
eat_event (ClutterActor *stage, ClutterEvent *event)
{
  return TRUE;
}

void      
launcher_util_stop_input ()
{
  if (!eat)
  {
    eat = clutter_rectangle_new ();
    clutter_container_add_actor (CLUTTER_CONTAINER(clutter_stage_get_default()),
                                 eat);
    clutter_actor_set_reactive (eat, TRUE);
    clutter_actor_set_opacity (eat, 0);
    clutter_actor_show (eat);
  }
  clutter_grab_pointer (eat);
  eat_tag = g_signal_connect (eat, "button-release-event", 
                              G_CALLBACK (eat_event), NULL);
}

void
launcher_util_start_input ()
{
  clutter_ungrab_pointer ();
  if (eat_tag) g_signal_handler_disconnect (eat, eat_tag);
  eat_tag = 0;
}
